﻿using System.Data;
using Framework.Data;

namespace Framework.Web
{
    /// <summary>
    /// リストボックスの選択候補、コードから名称への変換などで使用するリストデータの管理/取得を行うクラスです。
    /// リストのデータソースはXMLファイル、またはデータベースなどが利用できます。
    /// XMLファイルをデータソースにする場合は、ListMgr.ListSrcXMLFilePath　にパスを指定、その他のデータソースを使用する場合は、
    /// リスナを介してアプリケーション側へリクエストするので、アプリケーション起動時などでリスナの登録が必要です。
    /// ListMgr.SetListener(リスナのインスタンス)。
    /// </summary>
    public static class ListMgr
    {
        /// <summary>
        /// データソースにXMLファイルを使用する場合のファイルパス
        /// </summary>
        public static string ListSrcXMLFilePath { get; set; }

        private static IListSrcRequestListener _listener;

        private const string CACHE_KEY = "Framework.Web.ListSrc.Cache";
        private const string CACHE_KEY2 = "Framework.Web.ListSrc.DataView";

        private static ListSrcDataSet.ListSrcDataTable ListSrc
        {
            get
            {
                var ctx = System.Web.HttpContext.Current;

                var src = ctx.Cache[CACHE_KEY] as ListSrcDataSet;

                if (src == null)
                {
                    src = new ListSrcDataSet();
                    System.Web.Caching.CacheDependency cd = null;

                    if (!string.IsNullOrEmpty(ListMgr.ListSrcXMLFilePath))
                    {
                        src.ListSrc.ReadXml(ListMgr.ListSrcXMLFilePath);
                        cd = new System.Web.Caching.CacheDependency(ListMgr.ListSrcXMLFilePath);
                    }
                    else if (_listener != null)
                    {
                        var arg = new ListSrcRequestArgs();
                        arg.ListSrc = src;
                        arg.CacheDependency = null;
                        _listener.RequestListSrc(arg);

                        src = arg.ListSrc;
                        cd = arg.CacheDependency;
                    }
                    src.ListSrc.DefaultView.Sort = "ListKey, Sort";
                    
                    ctx.Cache.Insert(CACHE_KEY, src, cd);

                    var dv2 = new DataView(src.ListSrc);
                    dv2.Sort = "ListKey, Value";
                    ctx.Cache.Insert(CACHE_KEY2, dv2);
                }

                return src.ListSrc;
            }
        }

        private static DataView ListSrcView
        {
            get { return ListMgr.ListSrc.DefaultView; }
        }

        private static DataView ListSrcView2
        {
            get 
            {
                var ctx = System.Web.HttpContext.Current;

                var dv = ctx.Cache[CACHE_KEY2] as DataView;
                if (dv == null)
                {
                    var listSrc = ListSrc;
                    dv = ctx.Cache[CACHE_KEY2] as DataView;
                }
                return dv; 
            }
        }

        /// <summary>
        /// キャッシュをクリアします。データは次回アクセス時に再作成されます。
        /// </summary>
        public static void ClearCache()
        {
            var ctx = System.Web.HttpContext.Current;

            ctx.Cache.Remove(CACHE_KEY);
            ctx.Cache.Remove(CACHE_KEY2);
        }

        /// <summary>
        /// listKeyに該当するリストソースを返します。
        /// </summary>
        /// <param name="listKey"></param>
        /// <returns></returns>
        public static ListSrcDataSet GetListSrc(string listKey)
        {
            var ds = new ListSrcDataSet();

            foreach (DataRowView row in ListSrcView)
            {
                if (row["ListKey"].ToString() == listKey)
                {
                    ds.ListSrc.Rows.Add(row.Row.ItemArray);
                }
            }

            return ds;
        }

        /// <summary>
        /// 指定したlistKey、valueに該当するリストソースの行データを返します。見つからない場合はnullを返します。
        /// </summary>
        /// <param name="listKey"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public static ListSrcDataSet.ListSrcRow GetListSrcRow(string listKey, string value)
        {
            ListSrcDataSet.ListSrcRow ret = null;

            var rows = ListSrcView2.FindRows(new object[] { listKey, value });
            if (rows != null && rows.Length > 0)
            {
                ret = rows[0].Row as ListSrcDataSet.ListSrcRow;
            }

            return ret;
        }

        /// <summary>
        /// 指定したlistKey、valueに該当するリストソースの行データのTextを返します。
        /// </summary>
        /// <param name="listKey"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public static string GetText(string listKey, string value)
        {
            string ret = null;

            var row = GetListSrcRow(listKey, value);
            if (row != null)
            {
                ret = row.Text;
            }

            return ret;
        }

        /// <summary>
        /// 指定した引数でListSrcを作成します。
        /// </summary>
        /// <param name="items">_1=key, _2=value</param>
        /// <returns></returns>
        public static ListSrcDataSet CreateListSrc(System.Collections.Generic.IEnumerable<Tuple<string, string>> items)
        {
            var listSrc = new ListSrcDataSet();
            foreach (var item in items)
            {
                listSrc.ListSrc.AddListSrcRow(string.Empty, item._1, item._2, 1);
            }
            return listSrc;
        }

        /// <summary>
        /// 指定した引数でListSrcを作成します。
        /// </summary>
        /// <param name="items">_1=key, _2=value</param>
        /// <param name="lang">ローカライズの言語。value値をキーにローカライズし、その結果をvalueとする。</param>
        /// <returns></returns>
        public static ListSrcDataSet CreateListSrc(System.Collections.Generic.IEnumerable<Tuple<string, string>> items, string lang)
        {
            var listSrc = new ListSrcDataSet();
            foreach (var item in items)
            {
                listSrc.ListSrc.AddListSrcRow(string.Empty, item._1, LTextMgr.GetText(lang, item._2), 1);
            }
            return listSrc;
        }

        /// <summary>
        /// リストソースのリクエストリスナーを登録します。
        /// </summary>
        /// <param name="listener"></param>
        public static void SetListener(IListSrcRequestListener listener)
        {
            _listener = listener;
        }
    }
}
